/**
 * \file: mspin_connection_tcp_manager.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN TLS/TCP Connection Manager
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
* \copyright: (c) 2003 - 2016 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#ifndef MSPIN_CONNECTION_TCP_TLS_MANAGER_H_
#define MSPIN_CONNECTION_TCP_TLS_MANAGER_H_

#include "mspin_typedef.h"
#include "mspin_appl_if_adapter.h"

#include <openssl/ssl.h>    //SSL_CTX
#include <netdb.h>          //getaddrinfo, in_addr_t

typedef enum
{
    MSPIN_TLS_CONNECTED = 0,            //!< TLS connection verified and accepted
    MSPIN_TLS_SETUP_ERROR = -1,         //!< A setup error (e.g. SSL handle is NULL)
    MSPIN_TLS_CONNECTION_ERROR = -2,    //!< A connection failure. E.g. connection not found
    MSPIN_TLS_ACCEPT_ERROR = -3,        //!< TLS_Accept returned an error
    MSPIN_TLS_VERIFICATION_ERROR = -4,  //!< TLS verification failed
    MSPIN_TLS_NO_CLIENT_CERT = -5,      //!< Client did not provide a certificate
} mspin_tls_accept_result_t;

typedef enum
{
    MSPIN_CONNECTION_CLOSED = 0,     //!< socket connection is not in use or closed
    MSPIN_CONNECTION_HANDSHAKE = 1,  //!< socket connection is open, TLS handshake is in progress
    MSPIN_CONNECTION_ACCEPTED = 2,   //!< socket connection is open and TLS handshake is done if applicable
    MSPIN_CONNECTION_ACTIVE = 3      //!< socket connection is open and in use
} mspin_connection_state_t;

typedef struct
{
    int socketFD;
    mspin_connection_state_t state;
    in_addr_t ipAddr;
    SSL* pSSL;
    pthread_mutex_t mutex;
    bool error;
    char macAddr[18];
#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
    pthread_t threadID;         //thread ID of TCP reader
    mqd_t mesgQueue;
    bool quitReader;            //quit reader flag
#endif //#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
} mspin_tcp_connection_t;

const char* mspin_tcp_getReasonName(MSPIN_TCP_CONNECTION_END_REASON reason);

MSPIN_ERROR mspin_tcp_createNewConnection(const int socketFD, const in_addr_t ipAddr);

in_addr_t mspin_tcp_getIPAddr(const int socketFD);

MSPIN_ERROR mspin_tcp_getStoredMacAddress(const int socketFD, char* macAddress);

bool mspin_tcp_setActive(const int socketFD);

bool mspin_tcp_setAccepted(const int fd);

bool mspin_tcp_getError(const int socketFD);

bool mspin_tcp_setError(const int socketFD, bool error);

SSL* mspin_tcp_lockSSL(const int socketFD);

void mspin_tcp_unlockSSL(const int socketFD);

mspin_tls_accept_result_t mspin_tls_acceptConnection(const int socketFD, SSL_CTX* pSSLContext, bool verifyClient);

MSPIN_ERROR mspin_tcp_closeConnection(const int socketFD);

MSPIN_ERROR mspin_tcp_startWatching(void);

MSPIN_ERROR mspin_tcp_stopWatching(void);

void mspin_tcp_signalNewConnection(void);

#endif /* MSPIN_CONNECTION_TCP_TLS_MANAGER_H_ */
